## Python 3.10
"""
Copyright (c) 2023 Kohei "Max" MATSUSHITA (ma2shita+git@ma2shita.jp)
Released under the MIT license
https://opensource.org/licenses/mit-license.php
"""

"""Requires Envrionments
- DEST_AWS_ACCESS_KEY_ID (from IAM User)
- DEST_AWS_SECRET_ACCESS_KEY (from IAM User)
- DEST_ENDPOINT_URL (from IoT Core.)
- DEST_REGION_NAME (DEST_ENDPOINT_URL's region name.)
- LINE_CHANNEL_SECRET (from LINE Developers / Message API)
- LINE_CHANNEL_ACCESS_TOKEN (from LINE Developers / Message API)
"""

"""test:linebot-incoming
{
  "headers": {
    "x-line-signature": "dALOTAxJ5jXaOnSKx7IfSI5MMrT40375tAyzQSmmKnQ="
  },
  "pathParameters": {
    "channel_id": "9876543210"
  },
  "requestContext": {
    "stage": "test-invoke-stage"
  },
  "body": "{\"events\":[{\"replyToken\":\"DUMMY\",\"message\":{\"text\":\"90 test msg!\"}}]}"
}
"""

import json
import os

IS_TESTING_OR_DEVELOPMENT = True

def lambda_handler(event, context):
    print(f'# {event=}')
    global IS_TESTING_OR_DEVELOPMENT
    IS_TESTING_OR_DEVELOPMENT = is_testing_or_development_check(event)

    # get channel secret and check x-line-signature
    line_channel_secret = get_line_channel_secret()
    print(f'# {line_channel_secret=}')
    body = event['body']
    sig = event['headers']['x-line-signature']
    if not check_x_line_signature(line_channel_secret, body, sig):
        print('## Does not match x-line-signature, done.')
        return {'statusCode': 403, 'body': 'Does not match x-line-signature'}
    
    # finding reply token in event
    reply_token = find_reply_token_in(event)
    print(f'# {reply_token=}')

    # extract message
    try:
        text = json.loads(event['body'])['events'][0]['message']['text']
    except (IndexError, KeyError) as _:
        print('## Not exists `events[0].message.text`, done.')
        return {'statusCode': 200, 'body': 'Not exists `events[0].message.text`, done.'}

    # capture from message
    import re
    m = re.match('^([+-]?\d{1,3})\s+(.+)$', text)
    if not m:
        reply_to_line(reply_token, help_message())
        print('## Does not match message format, done.')
        return {'statusCode': 200, 'body': 'Does not match message format, done.'}
    
    # Building payload for IoT Core's device shadow doc.
    payload = { "state": { "desired": {
        "degree": m[1],
        "message": m[2].strip()[:12] # cap = 12 length
    }}}
    print(f'# {payload=}')

    # fire !!
    import boto3
    iot = boto3.client('iot-data',
        region_name=os.getenv('DEST_REGION_NAME'),   # us-west-2
        endpoint_url=os.getenv('DEST_ENDPOINT_URL'), # https://****-ats.iot.us-west-2.amazonaws.com
        aws_access_key_id=os.getenv('DEST_AWS_ACCESS_KEY_ID'), # AKI.. (Required `Allow iot:Publish`)
        aws_secret_access_key=os.getenv('DEST_AWS_SECRET_ACCESS_KEY') # *****
    )
    iot.update_thing_shadow(thingName="mcu1", shadowName="keiganmotor", payload=json.dumps(payload, ensure_ascii=False))
    reply_to_line(reply_token, accept_message())
    
    return {'statusCode': 200, 'body': 'Completed.'}
    
    """refs:
    - [Publish to IoT Endpoint of other Account - IoTData SDK NodeJS Lambda](https://repost.aws/questions/QUyQP-7Ki6T2202ZHCZ-qkig/publish-to-io-t-endpoint-of-other-account-io-t-data-sdk-node-js-lambda)
    - [Boto3の設定あれこれ（profile名、アクセスキー、Config、DEBUGログの設定とか）](https://zenn.dev/sugikeitter/articles/how-to-use-boto3-various-settings)
    - [Boto3 / Session reference](https://boto3.amazonaws.com/v1/documentation/api/latest/reference/core/session.html)
    - [Boto3 / IoTDataPlane](https://boto3.amazonaws.com/v1/documentation/api/latest/reference/services/iot-data.html)
    - [LINE Bot を5分で作ってみる (API Gateway 編)](https://blog.yuu26.com/line-bot-with-api-gateway/)
    - https://regex101.com/
    """

def is_testing_or_development_check(event):
    """Check working environment
    
    Args:
        event (dict): `event` from handler of Lambda function.
    
    Returns:
        bool: True is "Testing environment"
    
    Note:
        Amazon API Gateway のテスト環境からの呼び出しは
        requestContext.stage が "test-invoke-stage" となるので、それを利用しています。
    """
    r = event['requestContext']['stage'] == 'test-invoke-stage'
    print(f'## is_testing_or_development_check: {r}')
    return r

def check_x_line_signature(line_channel_secret, body_in_payload, x_line_signature):
    """Checking x-line-signature

    Args:
        channel_secret (str): "Channel Secret" in Messaging API console.
        body_in_payload (str): "body" in the payload sent by the LINE Messaging API webhook.
        x_line_signature (str): HTTP header sent by the LINE Messaging API webhook.

    Returns:
        bool: True if the signatures match.

    Examples:
        # In case of Amazon API Gateway
        cs = 'CHANNEL_SECRET'
        body = event['body']
        sig = event['headers']['x-line-signature']
        if (check_x_line_signature(cs, body, sig)):
            return {'statusCode': 200, 'body': ''}
        else:
            return {'statusCode': 403, 'body': ''}
    
    Note:
        ref: https://developers.line.biz/ja/docs/messaging-api/receiving-messages/#verifying-signatures
    """
    import base64
    import hashlib
    import hmac
    h = hmac.new(
            line_channel_secret.encode('utf-8'),
            body_in_payload.encode('utf-8'),
            hashlib.sha256).digest()
    s = base64.b64encode(h)
    i = x_line_signature.encode('utf-8')
    print(f'## check_x_line_signature: digest is {s}, `x-line-signature` is {i}')
    return s == i

def find_reply_token_in(event):
    """Find "reply token" in event
    
    Args:
        event (dict): `event` from handler of Lambda function.
    
    Returns:
        str: Reply token if found
        None: Not exists
    """
    body = json.loads(event['body'])
    try:
        reply_token = body['events'][0]['replyToken']
    except (IndexError, KeyError) as _:
        reply_token = None
    return reply_token

def reply_to_line(reply_token, reply_message):
    """Reply to LINE using Messaging API
    
    Args:
        reply_token (str or None): In message event payload.
        message (str): Reply message.
        is_testing (bool): True is "Testing"
    
    Returns:
        None.
    
    Note:
        `type=text` のみサポートしています。`reply_token` が None もしくは
        `is_testing` が True ならば送信しません。
        Bearer に使用するチャネルアクセストークンは `get_line_channel_access_token()`
        から取得しています。
    """
    global IS_TESTING_OR_DEVELOPMENT
    if IS_TESTING_OR_DEVELOPMENT:
        print('## reply_to_line: SKIP due to Testing')
        return
    if reply_token is None:
        print('## reply_to_line: SKIP due reply_token is None')
    ca_token = get_line_channel_access_token()
    if ca_token is None:
        print(f'## reply_to_line: SKIP due to cannot fetch LINE_CHANNEL_ACCESS_TOKEN')
        return
    headers = {
        'Content-Type': 'application/json',
        'Authorization': f'Bearer {ca_token}'
    }
    body = {
        'replyToken': reply_token,
        'messages': [
            {
                'type': 'text',
                'text': reply_message
            }
        ]
    }
    url = 'https://api.line.me/v2/bot/message/reply'
    import urllib.request
    req = urllib.request.Request(url, data=json.dumps(body).encode('utf-8'), method='POST', headers=headers)
    with urllib.request.urlopen(req) as res:
        r = res.read().decode('utf-8')
        print(f'## reply_to_line: {r}')

def help_message():
    return """\
中級の皆様、こんにちは！
オブジェクトを回転させるには、以下のフォーマットでメッセージを送ってね！
[角度 -359 ~ 359 ] [メッセージ]
例)
# 左回りに90度、ディスプレイに "Left!!" と表示
90 Left!!
# 右回りに180度、ディスプレイに "Right...!" と表示
-180 Right...!
# 左回りにグルっと一回転、ディスプレイに "I'm Max"
360 I'm Max
※メッセージは半角英数12文字まで。日本語や顔文字はNG。"""

def accept_message():
    return """\
受け付けたよ！確認してみてね (^^)v"""

def get_line_channel_secret():
    global IS_TESTING_OR_DEVELOPMENT
    if IS_TESTING_OR_DEVELOPMENT:
        print('## get_line_channel_secret: Using TEST Channel Secret')
        return 'CHANNEL_SECRET_FOR_TEST'
    return os.environ['LINE_CHANNEL_SECRET']

def get_line_channel_access_token():
    return os.environ['LINE_CHANNEL_ACCESS_TOKEN']
